'use client'

import { useEffect, useState } from 'react'

const Speech = () => {
    const [text, setText] = useState('')
    const [isSpeaking, setIsSpeaking] = useState(false)
    const [voices, setVoices] = useState([])
    const [selectedVoice, setSelectedVoice] = useState(null)

    useEffect(() => {
        const loadVoices = () => {
            const availableVoices = window.speechSynthesis.getVoices()
            setVoices(availableVoices)

            // Automatically select an Indian Hindi voice if available
            const hindiVoice = availableVoices.find(
                voice => voice.lang === 'hi-IN',
            )
            setSelectedVoice(hindiVoice || availableVoices[0])
        }

        if (typeof window !== 'undefined' && 'speechSynthesis' in window) {
            loadVoices()
            window.speechSynthesis.onvoiceschanged = loadVoices
        }
    }, [])

    const speak = () => {
        if (!text.trim() || !selectedVoice) return

        const utterance = new SpeechSynthesisUtterance(text)
        utterance.voice = selectedVoice
        utterance.lang = 'hi-IN' // Hindi language code

        utterance.onstart = () => setIsSpeaking(true)
        utterance.onend = () => setIsSpeaking(false)

        window.speechSynthesis.speak(utterance)
    }

    const stopSpeaking = () => {
        if (isSpeaking && 'speechSynthesis' in window) {
            window.speechSynthesis.cancel()
            setIsSpeaking(false)
        }
    }

    return (
        <div
            style={{
                maxWidth: '500px',
                margin: '0 auto',
                textAlign: 'center',
            }}>
            <textarea
                rows="5"
                placeholder="यहां अपना हिंदी पाठ दर्ज करें..."
                value={text}
                onChange={e => setText(e.target.value)}
                style={{
                    width: '100%',
                    padding: '10px',
                    fontSize: '16px',
                    fontFamily: 'Arial, sans-serif',
                }}
            />
            <br />
            <select
                onChange={e =>
                    setSelectedVoice(
                        voices.find(v => v.name === e.target.value),
                    )
                }
                value={selectedVoice?.name || ''}
                style={{
                    width: '100%',
                    padding: '10px',
                    margin: '10px 0',
                    fontSize: '16px',
                }}>
                {voices
                    .filter(voice => voice.lang === 'hi-IN') // Filter Hindi voices
                    .map(voice => (
                        <option key={voice.name} value={voice.name}>
                            {voice.name} ({voice.lang})
                        </option>
                    ))}
            </select>
            <br />
            <button
                onClick={speak}
                disabled={isSpeaking || !selectedVoice}
                style={{
                    margin: '10px',
                    padding: '10px 20px',
                    fontSize: '16px',
                    backgroundColor: isSpeaking ? 'gray' : 'blue',
                    color: 'white',
                    border: 'none',
                    cursor: isSpeaking ? 'not-allowed' : 'pointer',
                }}>
                {isSpeaking ? 'बोल रहा है...' : 'बोलो'}
            </button>
            <button
                onClick={stopSpeaking}
                style={{
                    margin: '10px',
                    padding: '10px 20px',
                    fontSize: '16px',
                    backgroundColor: 'red',
                    color: 'white',
                    border: 'none',
                    cursor: 'pointer',
                }}>
                रोकें
            </button>
        </div>
    )
}

export default Speech
